
/* ------------------------------------------------------------------------- */
/* i2c-algo-smbus-ali.c i2c driver algorithms + driver for ali adapters	     */
/* ------------------------------------------------------------------------- */
/*   Copyright (C) 1995-1997 Simon G. Vogl
                   1998-2000 Hans Berglund

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		     */
/* ------------------------------------------------------------------------- */

/* With some changes from Kysti Mlkki <kmalkki@cc.hut.fi> and 
   Frodo Looijaard <frodol@dds.nl> ,and also from Martin Bailey
   <mbailey@littlefeet-inc.com> */

/* $Id: i2c-smb-ali.c,v 1.3 2001/01/09 20:48:43 slee Exp $ */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/malloc.h>
#include <linux/version.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <linux/ioport.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/pci.h>
#include <asm/io.h>

#include <linux/i2c-new.h>

/* ----- global defines ----------------------------------------------- */
#define DEB(x) if (debug>=1) x
#define DEB2(x) if (debug>=2) x
#define DEB3(x) if (debug>=3) x /* print several statistical values*/
#define DEBPROTO(x) if (debug>=9) x;
 	/* debug the protocol by showing transferred bits */
#define DEF_TIMEOUT (HZ)

/* SMB Registers */
#define SMBSTS	0
#define SMBCMD	1
#define STRT_PRT 2
#define SMBADDR	3
#define SMBDATAA 4
#define SMBDATAB 5
#define SMBBLOCK 6
#define SMBCMND 7	/* index */

//REG0 SMBSTS
#define SMB_S_TERMINATE 	0x80
#define SMB_S_BUS_COLLI 	0x40
#define SMB_S_DEVICE_ERR 	0x20
#define SMB_S_SMI_I_STS 	0x10
#define SMB_S_HST_BSY 		0x08
#define SMB_S_IDL_STS 		0x04
#define SMB_S_HSTSLV_STS 	0x02
#define SMB_S_HSTSLV_BSY 	0x01

//REG1 SMBCMD
#define SMB_TERMINATE 	0x04
#define SMB_T_OUT_CMD 	0x08

#define SMB_QUICK_CMD 	0x00
#define SMB_SNDRCVB 	0x10
#define SMB_RDWRB 	0x20
#define SMB_RDWRW 	0x30
#define SMB_RDWRBLK 	0x40

#define SMB_BLK_CLR 	0x80

#define ALI_VENDORID	0x10b9
#define ALI_BRG_DEVID	0x1533
#define ALI_PMU_DEVID	0x7101


/* ----- global variables ---------------------------------------------	*/

/* module parameters:
 */
static int debug=0;
static int bus_test=0;	/* see if the line-setting functions work	*/
static int bus_scan=0;	/* have a look at what's hanging 'round		*/
static unsigned char  irq=0;
static unsigned int  piobase=0;
static struct pci_dev *brgdev = NULL, *pmudev = NULL;
#if (LINUX_VERSION_CODE < 0x020301)
static struct wait_queue *smb_ali_wait = NULL;
#else
static wait_queue_head_t smb_ali_wait;
#endif

/* I2C bus is not sharable or queueable */
/* So there's only one outstanding transaction */
/* hence the interrupt handling is also simple */
/* mutual exclusion to i2c device is handled by i2c-core */
void smb_ali_interrupt(int irq, void *data, struct pt_regs *regs)
{
	DEB(printk("INT\n"););
        wake_up_interruptible(&smb_ali_wait);
}

static int smb_ali_init (void)
{
	unsigned char temp;
	int dev2irq[]={0,9,3,10,4,5,7,6,1,11,0,12,0,14,0,15};
	int irq2dev[]={0,8,0,2,4,5,7,6,0,1,3,9,11,0,13,15};

	brgdev = pci_find_device(ALI_VENDORID, ALI_BRG_DEVID, brgdev);
	pmudev = pci_find_device(ALI_VENDORID, ALI_PMU_DEVID, pmudev);

	if (!brgdev || !pmudev) {
		printk("Device not found\n");
		return -1;
	}

	pci_write_config_dword(pmudev, 0xe0, 0x200001); /* SMB enable */

	if (irq) {	/* from modules argument */
		if (irq2dev[irq] == 0) {
			printk("irq %d unusable\n", irq);
			return -1;
		}
		printk("Using irq %d\n", irq);
		pci_write_config_byte(brgdev, 0x77, irq2dev[irq] | 0x10); /* SMBIR */
	} else {
		pci_read_config_byte(brgdev, 0x77, &irq); /* SMBIR */
		irq = dev2irq[irq & 0xf];
	}

	if (irq) {
		int ret;
		ret = request_irq(irq, smb_ali_interrupt, 0, "SMB_ALI", 0);
		if (ret) {
			printk("request_irq failed\n");
			return -1;
		}
#if (LINUX_VERSION_CODE >= 0x020301)
		init_waitqueue_head(&smb_ali_wait);
#endif
	} else {
		printk("interrupt disabled\n");
	}

	pci_read_config_dword(pmudev, 0x14, &piobase); /* SMB I/O base */

	piobase &= PCI_IO_RANGE_MASK;
printk("piobase %x\n", piobase);

	if (!piobase) {
		printk("ALi SMBus is not initialized\n");
		return -1;
	}

	pci_read_config_byte(pmudev, 0xe0, &temp);
	temp = 1;
	pci_write_config_byte(pmudev, 0xe0, temp);
	pci_read_config_byte(pmudev, 0xe0, &temp);

	pci_read_config_byte(pmudev, 0xe1, &temp);

	pci_read_config_byte(pmudev, 0xe2, &temp);
	temp = 0x30;
	pci_write_config_byte(pmudev, 0xe2, temp);
	pci_read_config_byte(pmudev, 0xe2, &temp);

	pci_read_config_byte(pmudev, 0x04, &temp);
	pci_read_config_byte(pmudev, 0x5b, &temp);

        outb(4, piobase + SMBCMD); //abort reset host controller

	return 0;

}

static int smb_ali_cleanup(void)
{
	unsigned char temp;

	if (irq)
		free_irq(irq, 0);
	pci_read_config_byte(pmudev, 0xe0, &temp);
	temp = 0; //disable interface
	pci_write_config_byte(pmudev, 0xe0, temp);
	pci_read_config_byte(pmudev, 0xe0, &temp);
	return 0;
}


/*
 * Sanity check for the adapter hardware - check the reaction of
 * the bus lines only if it seems to be idle.
 */
#if 0
static int test_bus(struct i2c_algo_pcf_data *adap, char *name) {
	return (0);
}
#endif

s32 smb_ali_xfer (struct i2c_adapter * adapter, u16 addr,
                           unsigned short flags,
                           char read_write, u8 command, int size,
                           union i2c_smbus_data * data)
{
	unsigned int temp, rwcmd;
	unsigned long tmout=jiffies + DEF_TIMEOUT;
	int retry=0;

again:

        do {
                outb(0xff, piobase + SMBSTS);
                temp = inb(piobase + SMBSTS);
                udelay(1);
        } while ( (jiffies < tmout) && !(temp & SMB_S_IDL_STS));

	if (jiffies >= tmout) {
		printk("I2C timeout stat=%x\n", temp);
		return -1;
	}

	switch(read_write) {
	case I2C_SMBUS_READ:
		addr = (addr << 1) + 1;
		break;
	case I2C_SMBUS_WRITE:
		addr = (addr << 1);
		break;
	default:
		printk("Unknown read/write command %x\n", read_write);
		return -1;
	}

        outb(addr, piobase + SMBADDR);

	switch(size) {
	case I2C_SMBUS_QUICK:
		rwcmd = SMB_QUICK_CMD;
		break;
	case I2C_SMBUS_BYTE:
		rwcmd = SMB_SNDRCVB;
		break;
	case I2C_SMBUS_BYTE_DATA:
		rwcmd = SMB_RDWRB;
		break;
	case I2C_SMBUS_WORD_DATA:
		rwcmd = SMB_RDWRW;
		break;
	default:
		printk("Unknown data size %x\n", size);
		return -1;
	}
		
        outb(rwcmd, piobase + SMBCMD);

//	if (size != I2C_SMBUS_BYTE && size != I2C_SMBUS_QUICK)
	if (size != I2C_SMBUS_QUICK)
		outb(command, piobase + SMBCMND);

	if (read_write == I2C_SMBUS_READ || size == I2C_SMBUS_QUICK) {
		outb(0xff, piobase + STRT_PRT);

		if (irq) {
			interruptible_sleep_on_timeout(&smb_ali_wait, DEF_TIMEOUT );
			temp = inb(piobase + SMBSTS);
			if (temp & (SMB_S_BUS_COLLI|SMB_S_DEVICE_ERR)) {
				return -1;
			}
		} else {	/* polling */
			tmout = jiffies + DEF_TIMEOUT;
			do {
				udelay(1);
				temp = inb(piobase + SMBSTS);
				if (temp & (SMB_S_BUS_COLLI|SMB_S_DEVICE_ERR)) {
					return -1;
				}
			} while ((jiffies < tmout) && !(temp & SMB_S_SMI_I_STS));
			if (temp == SMB_S_IDL_STS) {
				if (retry) {
					printk("SMB controller is gone...\n"
						"doesn't wake up.. give up\n");
					return -1; /* timeout */
				}
				printk("SMB controller is gone... resetting\n");
				outb(4, piobase + SMBCMD);
				udelay(10);
				retry = 1;
				goto again;
			}
			if (jiffies >= tmout) {
				return -1; /* timeout */
			}
		}
		if (size == I2C_SMBUS_QUICK) {
			return 1;
		}
		temp = inb(piobase + SMBDATAB);
		temp = temp << 8 | inb(piobase + SMBDATAA);
		data->word = temp;
		return 0;
	}
	else {
		if (size != I2C_SMBUS_BYTE) {
			outb(data->word>>8, piobase + SMBDATAB);
			outb(data->word, piobase + SMBDATAA);
		}

		outb(0xff, piobase + STRT_PRT);

		if (irq) {
			interruptible_sleep_on_timeout(&smb_ali_wait, DEF_TIMEOUT);
			temp = inb(piobase + SMBSTS);
			if (temp & (SMB_S_BUS_COLLI|SMB_S_DEVICE_ERR)) {
				return -1;
			}
		} else { /* polling */
			tmout = jiffies + DEF_TIMEOUT;
			do {
				udelay(1);
				temp = inb(piobase + SMBSTS);
				if (temp & (SMB_S_BUS_COLLI|SMB_S_DEVICE_ERR)) {
					return -1;
				}
			} while ((jiffies < tmout) && !(temp & SMB_S_SMI_I_STS));
			if (temp == SMB_S_IDL_STS) {
				if (retry) {
					printk("SMB controller is gone...\n"
						"doesn't wake up.. give up\n");
					return -1; /* timeout */
				}
				printk("SMB controller is gone... resetting\n");
				outb(4, piobase + SMBCMD);
				udelay(10);
				retry = 1;
				goto again;
			}
			if (jiffies >= tmout) {
				return -1; /* timeout */
			}
		}
		return 0;
	}

}

static int smb_control(struct i2c_adapter *adapter, 
	unsigned int cmd, unsigned long arg)
{
	return 0;
}

static u32 smb_func(struct i2c_adapter *adap)
{
	return I2C_FUNC_SMBUS_EMUL | I2C_FUNC_10BIT_ADDR | 
	       I2C_FUNC_PROTOCOL_MANGLING; 
}

/* -----exported algorithm data: -------------------------------------	*/

static struct i2c_algorithm smb_algo = {
	"SMB driver for ALi chipset",
	I2C_ALGO_SMBUS,
	NULL,
	smb_ali_xfer,
	NULL,				/* slave_xmit		*/
	NULL,				/* slave_recv		*/
	smb_control,			/* ioctl		*/
	smb_func,			/* functionality	*/
};

static struct i2c_adapter smb_ali = {
        "ALi chipset SMB adapter",
        I2C_HW_SMBUS_ALI15X3,
        NULL,
        NULL,
        NULL,
        NULL,
        NULL,
        NULL,
};


/* 
 * registering functions to load algorithms at runtime 
 */
int smb_add_bus(struct i2c_adapter *adap)
{
	int i, temp;

	if (bus_test) {
		//int ret = smb_test_bus(pcf_adap, adap->name);
		//if (ret<0)
		//	return -ENODEV;
	}

	DEB2(printk("i2c-smbus-ali.o: hw routines for %s registered.\n",
	            adap->name));

	/* register new adapter to i2c module... */

	adap->id |= smb_algo.id;
	adap->algo = &smb_algo;

	adap->timeout = 100;	/* default values, should	*/
	adap->retries = 3;	/* be replaced by defines	*/

#ifdef MODULE
//	MOD_INC_USE_COUNT;
#endif

	i2c_add_adapter(adap);

	/* scan bus */
	if (bus_scan) {
		printk(KERN_INFO " i2c-algo-smbus: scanning bus %s.\n", adap->name);
		for (i = 0x00; i < 0xff; i+=2) {
			unsigned long tmout = jiffies + HZ;
			do {
				outb(0xff, piobase + SMBSTS);
				temp = inb(piobase + SMBSTS);
				udelay(1);
			} while ((jiffies < tmout) && !(temp & SMB_S_IDL_STS));

			if (jiffies >= tmout) 
				continue;
			outb(i, piobase + SMBADDR);
			outb(SMB_QUICK_CMD, piobase + SMBCMD);
			outb(0xff, piobase + STRT_PRT);

			tmout = jiffies + HZ;
			do {
				temp = inb(piobase + SMBSTS);
				udelay(1);
			} while ((jiffies < tmout) && !(temp & (SMB_S_SMI_I_STS | SMB_S_BUS_COLLI | SMB_S_DEVICE_ERR)) );

			if (temp & SMB_S_SMI_I_STS) {
				printk("(%02x)",i>>1); 
			} else {
				printk("."); 
			}
		}
		printk("\n");
	}
	return 0;
}


int smb_del_bus(struct i2c_adapter *adap)
{
	i2c_del_adapter(adap);
	DEB2(printk("i2c-algo-smbus.o: adapter unregistered: %s\n",adap->name));

#ifdef MODULE
//	MOD_DEC_USE_COUNT;
#endif
	return 0;
}

int __init smb_init (void)
{
	printk("i2c-algo-smbus.o: i2c smbus ali algorithm module\n");
	return 0;
}


//EXPORT_SYMBOL(smb_add_bus);
//EXPORT_SYMBOL(smb_del_bus);

#ifdef MODULE
MODULE_AUTHOR("Soohoon Lee <soohoon.lee@alpha-processor.com>");
MODULE_DESCRIPTION("SMBUS algorithm for ALi chipset");

MODULE_PARM(irq, "i");
MODULE_PARM(bus_test, "i");
MODULE_PARM(bus_scan, "i");
MODULE_PARM(debug,"i");

MODULE_PARM_DESC(bus_test, "Test if the I2C bus is available");
MODULE_PARM_DESC(bus_scan, "Scan for active chips on the bus");
MODULE_PARM_DESC(debug, "debug level - 0 off; 1 normal; 2,3 more verbose");


int init_module(void) 
{
	int result = 0;

	result = smb_ali_init();

	if ( result < 0 ) {
		return -ENODEV;
	}
	smb_add_bus(&smb_ali);
	return 0;
}

void cleanup_module(void) 
{
	smb_del_bus(&smb_ali);
	smb_ali_cleanup();
}
#endif
